;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Variable and Breed declarations ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

globals
[
  ;; quick start instructions variables
  quick-start  ;; current quickstart instruction displayed in the quickstart monitor
  qs-item      ;; index of the current quickstart instruction
  qs-items     ;; list of quickstart instructions

  ;; variables related to the health of the turtles
  old-num-infected       ;; number of turtles that had infected? = true during the last pass through the go procedure
  current-num-infected   ;; number of turtles that have infected? = true during the current pass through the go procedure

  ;; list and string variables that hold data passed to or from calculators
  shape-names  ;; list that holds the names of the non-sick shapes a student's turtle can have
  time-data    ;; list that holds the times that turtles are infected
  sick-data    ;; list that holds current-num-infected for each pass through the go procedure when current-num-infected > old-num-infected
  who-sick     ;; string that holds the calculator userid's of all the turtles that are infected

  ;; misc
  clock        ;; keeps track of the number of times through the go procedure (if there is at least one turtle infected)
  color-changes-per-second  ;; the number of times the color of a student changes per second when it is flashing
  seconds-to-flash          ;; the number of seconds a student is supposed to flash
]

turtles-own
[
  immune?
  infected?    ;; if a turtle is ill, infected? is true, otherwise, it is false
  base-shape   ;; original shape of a turtle
  step-size    ;; the amount that a turtle will go forward in the current direction
]

breeds
[
  androids  ;; created by the CREATE ANDROIDS button; not controlled by anyone, but can become sick and spread sickness
  students  ;; created and controlled by the calculators
]

students-own
[
  user-id            ;; unique id generated by each calculator to identify each student or teacher turtle
  base-color         ;; the color that the turtle has originally
  flash?             ;; true if this turtle is flashing currently, false otherwise
  flashes-remaining  ;; the number of flashes this student still has to flash before it stops flashing
]


;;;;;;;;;;;;;;;;;;;;;
;; Setup Functions ;;
;;;;;;;;;;;;;;;;;;;;;

to startup
  setup-quick-start
  hubnet-set-client-interface "TI-83+" [ "AAA - Disease 2.3" [ "L1" ] ]
  hubnet-reset
  setup
end

;; Initializes the display, and creates a list that contains the names of the shapes
;; used by turtles in this activity.  The placement of the shape names in the last
;; corresponds to the numbers sent by calculators.  Also initializes the data lists.
to setup
  cp
  ct
  clear-output
  setup-vars
  setup-plot
  setup-hubnet
end

;; initialize global variables
to setup-vars
  set clock 0
  set color-changes-per-second 4
  set seconds-to-flash 5

  set old-num-infected 0
  set current-num-infected count turtles with [infected?]

  set shape-names ["wide wedge" "square" "car" "thin wedge" "big boat" "pickup truck"
                   "nu" "uu" "circle" "butterfly" "wolf" "sheep" "lobster" "monster"
                   "moose" "bear" "teddy bear"]
  set time-data []
  set sick-data []
  set who-sick " "
end

;; initialize the plot
to setup-plot
  clear-all-plots
  set-current-plot "Number Sick"
  set-current-plot-pen ("num-sick" + data-set)
  set-plot-y-range 0 (initial-number-sick + 5)
end

;; creates turtles that wander at random
to create-androids
  create-custom-androids num-androids
  [
    setxy (random-float screen-size-x) (random-float screen-size-y)
    set color gray
    set heading (random 4) * 90
    set infected? false
    ifelse random 100 < chance-immune [set immune? true] [set immune? false]
    set base-shape "android"
    set shape base-shape
    set step-size 1
  ]
end

;; give the user some information about what the setup button does so they can 
;; know whether they want to proceed before actually doing the setup
to setup-prompt
  if user-yes-or-no? ("The SETUP button should only be used when starting "
              + "over with a new group (such as a new set of students) since "
              + "all data is lost.  Use the RE-RUN button for continuing with "
              + "an existing group."
              + "\n\nDo you really want to setup the model?")
  [
    user-message "Before closing this dialog, please do the following:"
                 + "\n  -Have everyone that is currently logged in, log off and "
                 + "clear the calulator memory. (Press 2nd MEM 7 1 2)"
                 + "\n  -Open the teacher console for this activity and press the ERASE ALL DATA button."
    setup
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Functions ;;
;;;;;;;;;;;;;;;;;;;;;;;

to go
  ;; allow any turtles that are flashing to flash
  flash
  
  every 0.1
  [
    ;; get commands and data from the calculators
    listen-calc 

    ;;allow the androids to wander around the graphics window
    if wander?
    [ androids-wander ]

    check-infect
    
    ;; increment the clock if there are infected turtles and plot the data
    if current-num-infected > 0 and current-num-infected > old-num-infected
    [
      ;; if a turtle became sick during this clock tick, then
      ;; the data lists are updated and sent to the TISchool server
      set time-data lput clock time-data
      set sick-data lput current-num-infected sick-data
      send-info-to-calcs

      ;; plot the new data
      plotxy clock current-num-infected
      set old-num-infected current-num-infected
    ]
    if current-num-infected > 0
    [ set clock clock + 1 ]
  ]
end

;; makes any students that have flash? true flash 
;; (color-changes-per-second / 2) number of times 
;; each second for seconds-to-flash seconds
to flash
  if any? students with [ flash? ]
  [
    if any? students with [ flashes-remaining = 0 ]
    [
      ask students with [ flash? and flashes-remaining = 0 ]
      [
        set flash? false
        set color base-color
      ]
    ]
    if any? students with [ flashes-remaining > 0 ]
    [
      every (1 / color-changes-per-second)
      [
        ask students with [ flashes-remaining > 0 ]
        [
          ifelse (color = base-color)
          [ set color (red - 2) ]
          [ set color base-color ]
          set flashes-remaining flashes-remaining - 1
        ]
      ]
    ]
  ]
end

;; controls the motion of the androids
to androids-wander
  every android-delay
  [
    ask androids
    [
      rt (random 4) * 90
      fd move-amount
    ]
  ]
end

;; report the amount we can move in the current direction if we don't want to wrap around the screen
to-report move-amount  ;; turtle procedure
  locals
  [
    new-pcor  ;; the value of either pxcor or pycor depending upon if we are moving vertically or horizontally
    screen-edge  ;; the value of either screen-edge-x or screen-edge-y depending upon if we are moving vertically or horizontally
  ]
  ifelse heading mod 180 = 0
  [
    set screen-edge screen-edge-y
    ifelse heading = 0
    [ set new-pcor pycor + step-size ]
    [ set new-pcor pycor - step-size ]
  ]
  [
    set screen-edge screen-edge-x
    ifelse heading = 90
    [ set new-pcor pxcor + step-size ]
    [ set new-pcor pxcor - step-size ]
  ]
  ;; if we would step past a patch at the edge of the graphics window, make us step onto that patch
  if abs new-pcor > abs screen-edge
  [ report step-size - ( abs new-pcor - abs screen-edge ) ]
  report step-size
end

;; causes healthy turtles with an infected turtle on the same patch to test to see if they get sick
;; then have any infected turtles set their shape to the sick shape 
to check-infect
  ask turtles with [ not infected? and any? other-turtles-here with [ infected? ] 
  and not immune? ]
  [
      if ((random 100) + 1) <= percentage-infection
      [ get-sick ]
  ]
  ask turtles with [infected?]
  [ set-sick-shape ]

  ;; count the number of infected turtles each time, otherwise we could get more infected
  ;; turtles than turtles if an infected turtle dies and a new one is created later on
  set current-num-infected count turtles with [ infected? ]
end

;; turtle procedure -- set the appropriate variables to make this turtle sick
to get-sick
  set infected? true
  if (breed = students)
  [ set who-sick who-sick + " " + user-id ]
end

;; turtle procedure -- change the shape of turtles to its sick shape
;; if show-ill? is true and change the shape to the base-shape if
;; show-ill? is false
to set-sick-shape
  ifelse show-ill?
  [
    ;; we want to check if the turtles shape is already a sick shape
    ;; to prevent flickering in the turles
    if ( shape != ("sick " + base-shape) )
    [ set shape "sick " + base-shape ]
  ]
  [
    ;; we want to check if the turtles shape is already a base-shape
    ;; to prevent flickering in the turles
    if (shape != base-shape)
    [ set shape base-shape ]
  ]
end

;; causes the initial infection in the turtle population --
;; infects a random healthy turtle until the desired number of
;; turtles are infected
to infect-turtles
  locals
  [
    healthy-turtles ;; a list that holds the who values of all the turtles that are healthy
    doomed-turtle   ;; a random who from the healthy-turtles list
  ]

  set healthy-turtles values-from (turtles with [not infected? and not immune?]) [who]

  repeat initial-number-sick
  [
    ifelse length healthy-turtles = 0
    [
      user-message "There are no more healthy turtles to infect.  Infection stopped."
      stop
    ]
    [
      set doomed-turtle random-one-of healthy-turtles
      ask turtle doomed-turtle
      [ get-sick ]
      set healthy-turtles remove doomed-turtle healthy-turtles
    ]
  ]
  ask turtles with [ infected? ]
  [ set-sick-shape ]
end

;; heals all sick turtles,  clears and sets up the plot, 
;; and clears the lists sent to the calculators
to cure-all
  set clock 0
  ask turtles
  [
    set infected? false
    set shape base-shape
  ]

  set old-num-infected 0
  set current-num-infected 0

  ;; clear the calculator lists
  set time-data []
  set sick-data []
  set who-sick " "

  ;; clear and setup the plot
  setup-plot-for-rerun
  
  ;; setup hubnet again
  setup-hubnet
end

to setup-plot-for-rerun
  set-current-plot-pen ("num-sick" + data-set)
  plot-pen-reset
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Quick Start functions ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; instructions to quickly setup the model, calculators, and TISchool webpage to run this activity
to setup-quick-start
  set qs-item 0
  set qs-items
  [
    "Teacher: Follow these directions to run the HubNet activity."
    "Optional: Zoom In (see Tools in the Menu Bar)"
    "Optional: Change any of the settings...." "If you do change the settings, press the SETUP button."
    "Press the GO button."
    "Everyone: Reset the RAM on your calculator (2nd MEM 7 1 2)."
    "Login to your calculator."
    "Choose SIMULATION at the calculator Main Menu."
    "Remember your shape and color"
    "When you press enter again you will enter the simulation."
    "Teacher: Have the students move their turtles around to..." "acquaint themselves with the interface."
    "Press the INFECT NetLogo button to start the simulation."
    "Everyone: After some time of infections, choose..."
    "VIEW DATA at the calculator Main Menu to receive the data."
    
    "Teacher: To rerun the activity, do not clear the server."
    "Stop the simulation by pressing the NetLogo GO button."
    "Everyone: Exit to the calculator Main Menu."
    "Teacher: Change any of the settings that you would like."
    "To overlay plot data on the calculators..." "change the value of the slider CALCULATOR-DATA-SET."
    "If you set CALCULATOR-DATA-SET to a value that already..." "had data in it, the new data will over-write the old data."
    "Press the NetLogo RE-RUN button."
    "Everyone: Choose SIMULATION at the calculator Main Menu."
    "Teacher: Restart the simulation by pressing..." "the NetLogo GO button again."
    "Infect some turtles and continue."
    
    "Teacher: To start the simulation over with a new group,..." "stop the model by pressing the NetLogo GO button, if it is on."
    "Have everyone, including yourself, logout of their calculators."
    "Press the ERASE ALL DATA button on the TISCHOOL site."
    "Press the NetLogo SETUP button."
    "Follow these instructions from the beginning again."
  ]
  set quick-start (item qs-item qs-items)
end

;; view the next item in the quickstart monitor
to view-next
  set qs-item qs-item + 1
  if qs-item >= length qs-items
  [ set qs-item length qs-items - 1 ]
  set quick-start (item qs-item qs-items)
end

;; view the previous item in the quickstart monitor
to view-prev
  set qs-item qs-item - 1
  if qs-item < 0
  [ set qs-item 0 ]
  set quick-start (item qs-item qs-items)
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Code for interacting with the calculators ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; send initial settings to the calculators
to setup-hubnet
  ifelse show-ill-on-calcs?
  [ hubnet-broadcast "DISET" lput 1 list screen-edge-x screen-edge-y ]
  [ hubnet-broadcast "DISET" lput 0 list screen-edge-x screen-edge-y ]
  hubnet-broadcast "STR5" who-sick
end

;; determines which calculator sent a command, and what the command was
to listen-calc
  locals
  [
    current-data
    current-id
    cmd
  ]

  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    set current-id hubnet-message-source
    set current-data hubnet-message
    set cmd item 0 current-data
    execute-cmd current-id cmd current-data
  ]
end

;; Calculator command codes sent to NetLogo to indicate what each student turtle is supposed to be doing:
;; 0 - create and setup (set shape, color, and initial position) a student turtle
;; 1 - set current turtle position
;; 2 - make turtle flash to its sick shape
to execute-cmd [current-id cmd current-data]
  if cmd = 0
  [ exe-crt current-id current-data ]
  if cmd = 1
  [ exe-pos current-id current-data ]
  if cmd = 2
  [ exe-flash current-id ]
end

;; Command 0: Create a turtle, assign it to the current calculator,
;; and set its shape, color, and position
to exe-crt [current-id current-data]
  ifelse not any? students with [ user-id = current-id ]
  [
    create-custom-students 1
    [
      set user-id current-id
      setup-student-vars true
    ]
  ]
  [
    ask students with [ user-id = current-id ]
    [ setup-student-vars false ]
  ]
end

;; sets the turtle variables to appropriate initial values
to setup-student-vars [ full-setup? ] ;; turtle procedure
  if full-setup?
  [
    set infected? false
    set step-size 1
  ]
  ifelse random 100 < chance-immune [set immune? true] [set immune? false]
  set base-shape (item (item 1 hubnet-message) shape-names)
  ifelse infected?
  [ set-sick-shape ]
  [ set shape base-shape ]
  set base-color (item 2 hubnet-message)
  set color base-color
  setxy (item 3 hubnet-message) (item 4 hubnet-message)
  set heading random-one-of [0 90 270]
  set flash? false
  set flashes-remaining 0
end

;; Command 1: Set the active turtle's position
to exe-pos [ current-id current-data ]
  ask students with [ user-id = current-id ]
  [ setxy item 1 current-data item 2 current-data ]
end

;; Command 2: Cause the turtles to flash (to help students locate a particular turtle)
to exe-flash [ current-id ]
  ask students with [ user-id = current-id ]
  [
    set flash? true
    set flashes-remaining seconds-to-flash * color-changes-per-second
  ]
end

;; send back to the calculators the data collected from a single pass through the go function
to send-info-to-calcs
  hubnet-broadcast "STR5" who-sick
  hubnet-broadcast ("DATA" + calculator-data-set) sick-data
  hubnet-broadcast ("TIME" + calculator-data-set) time-data
end


; ***NetLogo Model Copyright Notice***

; This activity and associated models and materials was created as part of the project:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS.
; The project gratefully acknowledges the support of the 
; National Science Foundation ( REPP program ) -- grant number REC #9814682.

; Copyright 1999 by Uri Wilensky & Walter Stroup. Updated 2001,2002. All rights reserved.

; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from the copyright holders.
; Contact the copyright holders for appropriate licenses for redistribution for
; profit.

; To refer to this model in academic publications, please use:
; Wilensky, U. & Stroup, W. (1999).  NetLogo HubNet Disease model.
; http://ccl.northwestern.edu/netlogo/models/HubNetDisease.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.

; In other publications, please use:
; Copyright 1998 by Uri Wilensky and Walter Stroup.  All rights reserved.  See
; http://ccl.northwestern.edu/netlogo/models/HubNetDisease
; for terms of use.
;
; ***End NetLogo Model Copyright Notice***
@#$#@#$#@
GRAPHICS-WINDOW
272
96
645
490
5
5
33.0
1
10
1
1
1
0

CC-WINDOW
5
723
663
818
Command Center

BUTTON
2
10
62
43
Setup
setup-prompt
NIL
1
T
OBSERVER
T
NIL

BUTTON
130
10
194
43
Go
go
T
1
T
OBSERVER
T
NIL

SLIDER
34
191
231
224
percentage-infection
percentage-infection
0
100
100
1
1
%

BUTTON
195
10
253
43
Infect
infect-turtles
NIL
1
T
OBSERVER
T
NIL

BUTTON
6
285
115
318
Create Androids
create-androids
NIL
1
T
OBSERVER
T
NIL

PLOT
10
444
254
641
Number Sick
time
sick
0.0
25.0
0.0
6.0
true
false
PENS
"num-sick1" 1.0 0 -65536 true
"num-sick2" 1.0 0 -14592 true
"num-sick3" 1.0 0 -65281 true

SLIDER
117
318
264
351
android-delay
android-delay
0.0
10.0
0.0
0.1
1
NIL

SLIDER
117
285
264
318
num-androids
num-androids
1
200
42
1
1
NIL

SWITCH
160
238
261
271
show-ill?
show-ill?
0
1
-1000

SWITCH
6
318
115
351
wander?
wander?
0
1
-1000

MONITOR
255
10
654
59
Quick Start Instructions (more details in info tab)
quick-start
0
1

BUTTON
475
60
553
93
<<< PREV
view-prev
NIL
1
T
OBSERVER
T
NIL

BUTTON
552
60
636
93
NEXT >>>
view-next
NIL
1
T
OBSERVER
T
NIL

BUTTON
273
60
391
93
Reset Instructions
setup-quick-start
NIL
1
T
OBSERVER
T
NIL

MONITOR
9
230
67
279
turtles
count turtles
0
1

MONITOR
71
230
156
279
num-infected
current-num-infected
0
1

SLIDER
34
158
231
191
initial-number-sick
initial-number-sick
1
20
1
1
1
NIL

SLIDER
33
121
230
154
calculator-data-set
calculator-data-set
1
3
1
1
1
NIL

TEXTBOX
34
87
231
120
Slider controlling to which graph in the calculator data is passed

BUTTON
63
10
125
43
Re-Run
cure-all
NIL
1
T
OBSERVER
T
NIL

SWITCH
34
51
231
84
show-ill-on-calcs?
show-ill-on-calcs?
1
1
-1000

SLIDER
10
408
133
441
data-set
data-set
1
3
1
1
1
NIL

BUTTON
159
407
254
441
Clear Plot
setup-plot
NIL
1
T
OBSERVER
T
NIL

TEXTBOX
10
356
263
407
This slider controls which plot pen draws the data for the run.  This only changes the pen when Re-Run or Setup are pressed.

SLIDER
104
676
276
709
chance-immune
chance-immune
0
100
0
1
1
NIL

@#$#@#$#@
WHAT IS IT?
-----------
This model simulates the spread of a disease through a population.  This population can consist of either students, which are turtles controlled by individual students via the TI Navigator Network, or turtles that are generated and controlled by NetLogo, called androids, or both androids and students.

Turtles move around, possibly catching an infection.  Healthy turtles on the same patch as sick turtles have a PERCENTAGE-INFECTION chance of becoming ill.  A plot shows the number of sick turtles at each time tick, and if SHOW-ILL? is on, sick turtles have a red circle attached to their shape.

Initially, all turtles are healthy.  A number of turtles equal to INITIAL-NUMBER-SICK become ill when the INFECT button is depressed.

Whenever a new turtle becomes ill, data lists time-data and sick-data are sent to the TISchool server.  These are then accessible via the VIEW DATA option on the calculator menus.  The data is displayed in plots 1, 2, and 3, whose numbers correspond to the slider PLOT-NUMBER in the NetLogo model.

This activity requires the TI Navigator activity AAA - Disease 2.3

For further documentation, see the Participatory Simulations Guide found at http://ccl.northwestern.edu/ps/


HOW TO USE IT
-------------
IMPORTANT NOTE: Before running this with a class, be sure to press the ERASE ALL DATA button in the teacher console for the activity AAA - Disease 2.3 on the TISchool website.  If you choose to exit the program and return (using the USE CALC option at the Main Menu) or choose the RESTART option, *DO NOT* clear the server.

QUICKSTART INSTRUCTIONS:
------------------------
Contains instructions as to how to quickly setup the model, calculators, and TISchool web page so as to run this activity.  The instructions can be found below:

Teacher: Open the TISchool site in your web browser.
Enable activity AAA - Disease 2.3
Open the teacher console and press the ERASE ALL DATA button.

If it is not open already, open the NetLogo model.
If you are prompted by a Login dialog, enter your teacher id and password and any other necessary information.
Optional- Zoom In (see Tools in the Menu Bar)
Optional- Change any of the settings.  If you do change the settings, press the SETUP button.
Press the GO button.

Everyone: Reset the RAM on your calculator by pressing the following keys: 2nd MEM 7 1 2.
Login to your calculator.
Select the SIMULATION option from the calculator Main Menu.
The calculators will display the shape and color you will have in the NetLogo model.  Remember these so that you can find your turtle.  When you press enter again you will enter the simulation.

Teacher: Have the students move their turtles around to acquaint themselves with the interface.
When you are ready to simulate the spread of the disease, press the INFECT button in the NetLogo model.

Everyone: After some time of infections, choose VIEW DATA at the calculator Main Menu to receive the data that was collected by NetLogo.

Teacher: To rerun the activity, do not clear the server.
Stop the simulation by pressing the NetLogo GO button.
Everyone: Exit to the calculator Main Menu.
Teacher: Change any of the settings that you would like.  Now might be a good time to change to which set of data NetLogo sends information.  You do this by changing the value of the NetLogo CALCULATOR-DATA-SET slider.  If you set CALCULATOR-DATA-SET to a value that already had data in it, the new data will over-write the old data.
Press the NetLogo RE-RUN button.
Everyone: Choose SIMULATION at the calculator Main Menu.
Teacher: Once everyone has chosen SIMULATION from the Main Menu.
Restart the simulation by pressing the NetLogo GO button again.
Infect some turtles and continue.

Teacher: To start the simulation over with a new group, stop the model by pressing the NetLogo GO button, if it is on.
Have everyone, including yourself, logout of their calculators.
Press the ERASE ALL DATA button on the TISCHOOL site.
Press the NetLogo SETUP button.
Follow these instructions from the beginning again.


BUTTONS:
--------
SETUP - clears all turtles and patches and the plot.  This should only be pressed when starting out with a new group of users since all data is lost.
GO - runs the simulation
RE-RUN - cures all turtles, clears the plot, and clears the lists sent to the calculators.  This should be used to setup the model again for collecting more data or running the model again with the same users connected.  Every user should be in the calculator Main Menu when this is pressed.
CREATE ANDROIDS - adds randomly moving turtles to the simulation
INFECT - infects some of the turtles in the simulation
CLEAR PLOT - clears the plot
NEXT >>> - shows the next quick start instruction
<<< PREVIOUS - shows the previous quick start instruction
RESET INSTRUCTIONS - shows the first quick start instruction


SLIDERS:
--------
NUM-ANDROIDS - determines how many androids are created by the CREATE ANDROIDS button
ANDROID-DELAY - the delay time, in seconds, for android movement- the higher the number, the slower the androids move
INITIAL-NUMBER-SICK - the initial number of turtles to become infected when the INFECT button is pressed
PERCENTAGE-INFECTION - sets the percentage chance that every tenth of a second a healthy turtle will become sick if it is on the same patch as an infected turtle
CALCULATOR-DATA-SET - the number of the graph in the calculator to which the current data will be sent and display on the calculators when the VIEW DATA menu option is chosen
DATA-SET - determines the plot pen to draw the data with in the plot.  This only changes the plot pen when RE-RUN or SETUP are pressed.


SWITCHES:
---------
WANDER? - when on, the androids wander randomly.  When off, they sit still
SHOW-ILL? - when on, sick turtles add to their original shape a red circle.  When off, they can move through the populace unnoticed
SHOW-ILL-ON-CALCS? - when on, the calculators will show a label indicating if the turtle is sick or not.  This only is sent to the calculators when the RE-RUN or SETUP buttons are pressed.  The calculators receive this value when the SIMULATION option is chosen from the calculator Main Menu.


MONITORS:
---------
TURTLES - the number of turtles in the simulation
NUM-INFECTED - the number of turtles that are infected


PLOTS:
------
NUMBER SICK - shows the number of sick turtles versus time


CALCULATOR INFORMATION
----------------------

TEACHER CALCULATOR:
-------------------
Students and teacher have identical calculator programs.

STUDENT CALCULATOR:
-------------------
When the program runs, it checks the server for a "flag" to indicate whether the program is being restarted.  If the flag isn't found, then a splash screen will display saying "DISEASE SIMULATION" and variables will be initialized.  Then, the main menu will display.

At the main menu, you will find five options:

SIMULATION  - join the simulation
VIEW DATA   - view data from the simulation
RESTART     - restart, clearing all data
USE CALC    - use the built in functionality of the calculator
QUIT        - quit, clearing all data from the calculator

CALCULATOR MENU OPTIONS:
------------------------
SIMULATION:
When you choose to join the simulation, a message pops up letting you know what shape and color you will have in the NetLogo simulation.  This shape and color is randomly determined each time you run the activity, and you might share a shape and color with somebody else.  There are many possible shapes and colors for the turtles.

On the calculator a coordinate axis will appear, along with a small square representing the position of that student's turtle.  A softkey menu will also appear, with two options: BACK and STEP

Arrow keys - move the turtle
BACK - returns to the MAIN MENU
STEP - brings up a menu, where you can choose between steps of 1 thru 5.  Pressing the ENTER key also allows you to change the step-size.
MATH - makes your turtle flash red for while- this can help you find your turtle

VIEW DATA:
When you choose this option, data from the simulation is downloaded.  Up to three data sets can be collected and displayed as PLOT1, PLOT2, and PLOT3.  Initially, the plots are shown together, and you can toggle plots on and off using the middle three softkeys.  BACK returns to the MAIN MENU, and RFSH collects fresh data from the NetLogo model.

RESTART:
Restarts the activity without the need to rerun the programs.

USE CALC:
This allows you to temporarily leave the program, use the built in functionality of the calculator, and then return to the activity with your data preserved.  When you leave the program, the data lists from NetLogo are automatically placed in the stat editor.  To return to the program, run DISEASE on the calculator.

QUIT:
This quits the program, clearing all data and restoring your previous graphics settings.


THINGS TO NOTICE
----------------
No matter how you change the various parameters, the same basic plot shape emerges.  After using the model once with the students, ask them how they think the plot will change if you alter a parameter.  Altering the initial percentage sick and the percentage infection will have different effects on the plot.


THINGS TO TRY
-------------
Use the model with the entire class to serve as an introduction to the topic.  Then have students use the NetLogo model individually, in a computer lab, to explore the effects of the various parameters.  Discuss what they find, observe, and can conclude from this model.


EXTENDING THE MODEL
-------------------
Currently, the turtles remain sick once they're infected.  How would the shape of the plot change if turtles eventually healed?  If, after healing, they were immune to the disease, or could still spread the disease, how would the dynamics be altered?


CREDITS AND REFERENCES
----------------------
This activity and associated models and materials was created as part of the project:
PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS.
The project gratefully acknowledges the support of the National Science Foundation ( REPP program ) -- grant number REC #9814682.

Copyright 1999 by Uri Wilensky & Walter Stroup. Updated 2001,2002. All rights reserved.

Permission to use, copy, or modify this software and its associated documentation, models and curricular materials for educational and research purposes only and without fee is hereby granted, provided that this copyright notice and the original authors' names appear on all copies and supporting documentation. For any other uses of this software, in original or modified form, including, but not limited to distribution in whole or in part, specific prior permission must be obtained from Uri Wilensky and Walter Stroup. These programs shall not be used, rewritten, or adapted as the basis of a commercial or hardware product without first obtaining appropriate licenses from Uri Wilensky & Walter Stroup. We make no representations about the suitability of this software for any purpose. It is provided "as is" without express or implied warranty.

To refer to this model in academic publications, please use: Wilensky, U. & Stroup, W. (1999).  NetLogo HubNet Disease model. http://ccl.northwestern.edu/netlogo/models/HubNetDisease. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use: Copyright 1999 by Uri Wilensky and Walter Stroup.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/HubNetDisease for terms of use.
@#$#@#$#@
default
false
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

android
false
0
Rectangle -7566196 true true 105 74 210 239
Polygon -7566196 true true 104 78 34 129 47 148 114 89
Polygon -7566196 true true 198 81 274 108 258 142 192 104
Polygon -7566196 true true 115 239 115 289 133 289 133 237
Polygon -7566196 true true 176 235 176 287 192 287 192 234
Rectangle -7566196 true true 119 12 194 73
Rectangle -16777216 true false 129 22 147 36
Rectangle -16777216 true false 164 23 184 37
Rectangle -16777216 true false 151 113 163 125
Rectangle -16777216 true false 153 142 164 154
Rectangle -16777216 true false 154 171 166 184

bear
false
0
Rectangle -7566196 true true 106 35 181 92
Circle -7566196 true true 110 17 18
Circle -7566196 true true 128 17 19
Rectangle -7566196 true true 80 58 111 89
Rectangle -7566196 true true 98 95 101 95
Rectangle -7566196 true true 92 93 192 104
Rectangle -7566196 true true 83 101 199 114
Rectangle -7566196 true true 78 107 205 130
Rectangle -7566196 true true 74 122 209 240
Rectangle -7566196 true true 77 236 204 249
Rectangle -7566196 true true 106 89 180 104
Rectangle -7566196 true true 80 247 198 258
Rectangle -7566196 true true 85 255 108 281
Rectangle -7566196 true true 169 252 195 283
Rectangle -7566196 true true 158 276 171 283
Rectangle -7566196 true true 74 276 90 281
Circle -7566196 true true 188 201 35
Circle -16777216 true false 111 40 13
Rectangle -7566196 true true 51 163 87 184
Rectangle -7566196 true true 45 166 53 169
Rectangle -7566196 true true 42 172 52 173
Rectangle -7566196 true true 38 178 53 183
Circle -7566196 true true 72 62 20

big boat
false
0
Polygon -6524078 true false 1 196 43 296 193 296 297 194
Rectangle -1 true false 135 14 149 194
Polygon -7566196 true true 151 14 173 18 193 30 211 48 239 88 251 118 271 170 271 184 253 176 227 170 199 172 177 180 161 190 165 160 169 122 165 78
Polygon -7566196 true true 133 36 115 50 77 86 47 122 7 152 33 156 57 164 77 178 91 188
Rectangle -7566196 true true 30 206 234 220
Rectangle -7566196 true true 52 224 234 236
Rectangle -7566196 true true 78 240 234 250

butterfly
false
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7566196 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7566196 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7566196 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7566196 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7566196 true 167 47 159 82
Line -7566196 true 136 47 145 81
Circle -7566196 true true 165 45 8
Circle -7566196 true true 134 45 6
Circle -7566196 true true 133 44 7
Circle -7566196 true true 133 43 8

car
false
15
Circle -1 false true 185 55 60
Circle -1 false true 183 186 61
Polygon -1 true true 214 52 214 22 182 26 162 38 144 74 138 102 100 120 99 161 102 201 118 246 152 267 190 275 210 251 187 240 178 200 204 182 215 181 214 118 193 112 182 98 181 72 198 52

circle
false
0
Circle -7566196 true true 30 30 240

lobster
false
0
Polygon -7566196 true true 105 106 122 238 100 259 100 276 118 275 150 240 180 275 197 275 197 258 174 238 196 105 178 77 123 77 105 106
Polygon -7566196 true true 112 103 84 78 105 60 94 14 85 57 78 19 67 59 76 83 110 118
Polygon -7566196 true true 192 120 228 85 241 57 231 22 223 59 217 17 205 61 216 79 192 108
Rectangle -7566196 true true 71 125 117 133
Rectangle -7566196 true true 73 149 119 157
Rectangle -7566196 true true 78 178 123 187
Rectangle -7566196 true true 183 131 219 138
Rectangle -7566196 true true 180 152 216 160
Rectangle -7566196 true true 173 176 217 184
Rectangle -7566196 true true 127 56 136 82
Rectangle -7566196 true true 160 55 169 81

monster
false
0
Rectangle -7566196 true true 77 100 213 245
Rectangle -7566196 true true 189 243 213 273
Rectangle -7566196 true true 78 242 102 271
Rectangle -7566196 true true 59 119 83 239
Rectangle -7566196 true true 203 118 225 239
Rectangle -7566196 true true 225 123 264 158
Rectangle -7566196 true true 262 124 272 131
Rectangle -7566196 true true 263 133 273 140
Rectangle -7566196 true true 263 145 271 156
Rectangle -7566196 true true 19 122 61 156
Rectangle -7566196 true true 9 125 19 130
Rectangle -7566196 true true 11 135 20 140
Rectangle -7566196 true true 9 146 20 151
Rectangle -7566196 true true 89 79 198 101
Rectangle -7566196 true true 112 35 176 82
Rectangle -16777216 true false 122 41 136 53
Rectangle -16777216 true false 149 39 167 53
Circle -1 true false 82 118 117
Polygon -16777216 true false 122 62 132 77 157 77 168 62
Polygon -1 true false 140 63 135 72 127 62
Polygon -1 true false 145 61 151 70 159 62
Polygon -7566196 true true 123 17 115 34 131 34
Polygon -7566196 true true 165 15 149 33 172 34

moose
false
0
Polygon -7566196 true true 226 121 90 121 60 136 61 181 106 195 210 195 255 181 255 135
Rectangle -7566196 true true 61 180 75 268
Rectangle -7566196 true true 89 187 104 275
Rectangle -7566196 true true 210 186 225 269
Rectangle -7566196 true true 241 178 254 275
Polygon -7566196 true true 62 138 60 107 23 107 17 102 18 85 24 82 54 79 66 63 87 64 94 128
Circle -16777216 true false 63 72 12
Polygon -7566196 true true 226 120 278 138 284 185 265 188 260 142
Polygon -7566196 true true 65 65 70 50
Polygon -7566196 true true 68 64 60 49 17 49 7 41 6 12 22 40 38 40 35 14 49 39 66 39 64 16 78 66 68 64
Polygon -7566196 true true 76 64 97 17 95 38 111 38 122 16 121 38 136 38 152 11 151 38 144 47 97 47 86 66

nu
false
0
Rectangle -7566196 true true 16 31 60 285
Rectangle -7566196 true true 120 31 164 285
Rectangle -7566196 true true 182 31 226 285
Rectangle -7566196 true true 256 31 298 285
Rectangle -7566196 true true 214 241 274 285
Polygon -7566196 true true 60 31 118 209 120 281 60 105
Rectangle -7566196 true true 110 211 140 249

pickup truck
false
0
Polygon -7566196 true true 285 208 285 178 279 164 261 144 229 132 217 132 213 106 199 84 171 68 149 68 129 68 129 148 1 148 1 156 19 164 19 222 285 222 283 174 283 176
Circle -16777216 true false 40 185 71
Circle -16777216 true false 192 191 66
Circle -7566196 true true 195 194 59
Circle -7566196 true true 43 188 64
Polygon -16777216 true false 197 94 149 94 157 128 209 128 205 112 203 102 197 94
Polygon -7566196 true true 21 142 139 142 139 136 13 136

sheep
false
14
Rectangle -1 true false 90 75 270 225
Circle -1 true false 15 75 150
Rectangle -16777216 true true 81 225 134 286
Rectangle -16777216 true true 180 225 238 285
Circle -16777216 true true 1 88 92

sick android
false
0
Rectangle -7566196 true true 105 74 210 239
Polygon -7566196 true true 104 78 34 129 47 148 114 89
Polygon -7566196 true true 198 81 274 108 258 142 192 104
Polygon -7566196 true true 115 239 115 289 133 289 133 237
Polygon -7566196 true true 176 235 176 287 192 287 192 234
Rectangle -7566196 true true 119 12 194 73
Rectangle -16777216 true false 129 22 147 36
Rectangle -16777216 true false 164 23 184 37
Rectangle -16777216 true false 151 113 163 125
Rectangle -16777216 true false 153 142 164 154
Rectangle -16777216 true false 154 171 166 184
Circle -65536 true false 4 125 112

sick bear
false
0
Rectangle -7566196 true true 106 35 181 92
Circle -7566196 true true 110 17 18
Circle -7566196 true true 128 17 19
Rectangle -7566196 true true 80 58 111 89
Rectangle -7566196 true true 98 95 101 95
Rectangle -7566196 true true 92 93 192 104
Rectangle -7566196 true true 83 101 199 114
Rectangle -7566196 true true 78 107 205 130
Rectangle -7566196 true true 74 122 209 240
Rectangle -7566196 true true 77 236 204 249
Rectangle -7566196 true true 106 89 180 104
Rectangle -7566196 true true 80 247 198 258
Rectangle -7566196 true true 85 255 108 281
Rectangle -7566196 true true 169 252 195 283
Rectangle -7566196 true true 158 276 171 283
Rectangle -7566196 true true 74 276 90 281
Circle -7566196 true true 188 201 35
Circle -16777216 true false 111 40 13
Rectangle -7566196 true true 51 163 87 184
Rectangle -7566196 true true 45 166 53 169
Rectangle -7566196 true true 42 172 52 173
Rectangle -7566196 true true 38 178 53 183
Circle -7566196 true true 72 62 20
Circle -65536 true false 57 88 136

sick big boat
false
0
Polygon -6524078 true false 1 196 43 296 193 296 297 194
Rectangle -1 true false 135 14 149 194
Polygon -7566196 true true 151 14 173 18 193 30 211 48 239 88 251 118 271 170 271 184 253 176 227 170 199 172 177 180 161 190 165 160 169 122 165 78
Polygon -7566196 true true 133 36 115 50 77 86 47 122 7 152 33 156 57 164 77 178 91 188
Rectangle -7566196 true true 30 206 234 220
Rectangle -7566196 true true 52 224 234 236
Rectangle -7566196 true true 78 240 234 250
Circle -65536 true false 1 134 123

sick butterfly
false
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7566196 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7566196 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7566196 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7566196 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7566196 true 167 47 159 82
Line -7566196 true 136 47 145 81
Circle -7566196 true true 165 45 8
Circle -7566196 true true 134 45 6
Circle -7566196 true true 133 44 7
Circle -7566196 true true 133 43 8
Circle -65536 true false 167 99 128

sick car
false
15
Circle -1 false true 185 55 60
Circle -1 false true 183 186 61
Polygon -1 true true 214 52 214 22 182 26 162 38 144 74 138 102 100 120 99 161 102 201 118 246 152 267 190 275 210 251 187 240 178 200 204 182 215 181 214 118 193 112 182 98 181 72 198 52
Circle -65536 true false 19 167 82

sick circle
false
0
Circle -7566196 true true 35 35 230
Circle -65536 true false 101 104 98

sick lobster
false
0
Polygon -7566196 true true 105 106 122 238 100 259 100 276 118 275 150 240 180 275 197 275 197 258 174 238 196 105 178 77 123 77 105 106
Polygon -7566196 true true 112 103 84 78 105 60 94 14 85 57 78 19 67 59 76 83 110 118
Polygon -7566196 true true 192 120 228 85 241 57 231 22 223 59 217 17 205 61 216 79 192 108
Rectangle -7566196 true true 71 125 117 133
Rectangle -7566196 true true 73 149 119 157
Rectangle -7566196 true true 78 178 123 187
Rectangle -7566196 true true 183 131 219 138
Rectangle -7566196 true true 180 152 216 160
Rectangle -7566196 true true 173 176 217 184
Rectangle -7566196 true true 127 56 136 82
Rectangle -7566196 true true 160 55 169 81
Circle -65536 true false 90 95 118

sick monster
false
0
Rectangle -7566196 true true 77 100 213 245
Rectangle -7566196 true true 189 243 213 273
Rectangle -7566196 true true 78 242 102 271
Rectangle -7566196 true true 59 119 83 239
Rectangle -7566196 true true 203 118 225 239
Rectangle -7566196 true true 225 123 264 158
Rectangle -7566196 true true 262 124 272 131
Rectangle -7566196 true true 263 133 273 140
Rectangle -7566196 true true 263 145 271 156
Rectangle -7566196 true true 19 122 61 156
Rectangle -7566196 true true 9 125 19 130
Rectangle -7566196 true true 11 135 20 140
Rectangle -7566196 true true 9 146 20 151
Rectangle -7566196 true true 89 79 198 101
Rectangle -7566196 true true 112 35 176 82
Rectangle -16777216 true false 122 41 136 53
Rectangle -16777216 true false 149 39 167 53
Circle -1 true false 82 118 117
Polygon -16777216 true false 122 62 132 77 157 77 168 62
Polygon -1 true false 140 63 135 72 127 62
Polygon -1 true false 145 61 151 70 159 62
Polygon -7566196 true true 123 17 115 34 131 34
Polygon -7566196 true true 165 15 149 33 172 34
Circle -65536 true false 77 102 140

sick moose
false
0
Polygon -7566196 true true 226 121 90 121 60 136 61 181 106 195 210 195 255 181 255 135
Rectangle -7566196 true true 61 180 75 268
Rectangle -7566196 true true 89 187 104 275
Rectangle -7566196 true true 210 186 225 269
Rectangle -7566196 true true 241 178 254 275
Polygon -7566196 true true 62 138 60 107 23 107 17 102 18 85 24 82 54 79 66 63 87 64 94 128
Circle -16777216 true false 63 72 12
Polygon -7566196 true true 226 120 278 138 284 185 265 188 260 142
Polygon -7566196 true true 65 65 70 50
Polygon -7566196 true true 68 64 60 49 17 49 7 41 6 12 22 40 38 40 35 14 49 39 66 39 64 16 78 66 68 64
Polygon -7566196 true true 76 64 97 17 95 38 111 38 122 16 121 38 136 38 152 11 151 38 144 47 97 47 86 66
Circle -65536 true false 79 87 115

sick nu
false
0
Rectangle -7566196 true true 16 31 60 285
Rectangle -7566196 true true 120 31 164 285
Rectangle -7566196 true true 182 31 226 285
Rectangle -7566196 true true 256 31 298 285
Rectangle -7566196 true true 214 241 274 285
Polygon -7566196 true true 60 31 118 209 120 281 60 105
Rectangle -7566196 true true 110 211 140 249
Circle -65536 true false 110 147 121

sick pickup truck
false
0
Polygon -7566196 true true 285 208 285 178 279 164 261 144 229 132 217 132 213 106 199 84 171 68 149 68 129 68 129 148 1 148 1 156 19 164 19 222 285 222 283 174 283 176
Circle -16777216 true false 40 185 71
Circle -16777216 true false 192 191 66
Circle -7566196 true true 195 194 59
Circle -7566196 true true 43 188 64
Polygon -16777216 true false 197 94 149 94 157 128 209 128 205 112 203 102 197 94
Polygon -7566196 true true 21 142 139 142 139 136 13 136
Circle -65536 true false 22 111 66

sick sheep
false
14
Rectangle -1 true false 90 75 270 225
Circle -1 true false 15 75 150
Rectangle -16777216 true true 81 225 134 286
Rectangle -16777216 true true 180 225 238 285
Circle -16777216 true true 1 88 92
Circle -65536 true false 112 46 144

sick square
false
0
Rectangle -7566196 true true 45 45 255 255
Circle -65536 true false 100 97 102

sick teddy bear
false
0
Circle -7566196 true true 110 21 81
Circle -7566196 true true 97 7 39
Circle -7566196 true true 171 5 39
Polygon -7566196 true true 133 88 95 117 95 225 119 247 188 248 215 224 215 116 170 91
Circle -7566196 true true 134 238 34
Polygon -7566196 true true 197 113 249 88 261 91 268 106 262 116 205 139 197 113
Polygon -7566196 true true 115 116 66 90 54 93 45 110 50 117 103 145 115 116
Polygon -7566196 true true 104 204 54 233 54 244 63 257 71 256 117 227
Polygon -7566196 true true 194 228 240 255 248 254 260 238 257 231 204 207 194 228
Circle -1 true false 124 41 20
Circle -1 true false 158 42 20
Line -16777216 false 128 80 151 69
Line -16777216 false 151 69 174 83
Polygon -65536 true false 151 208 111 173 132 153 151 169 171 154 189 169

sick thin wedge
false
0
Polygon -7566196 true true 133 20 70 252 127 148 177 245
Circle -65536 true false 165 140 87

sick uu
false
0
Rectangle -7566196 true true 58 44 104 226
Rectangle -7566196 true true 88 178 194 224
Rectangle -7566196 true true 180 44 222 224
Rectangle -7566196 true true 122 74 168 284
Rectangle -7566196 true true 124 240 250 286
Rectangle -7566196 true true 240 76 284 286
Circle -65536 true false 43 208 93

sick wide wedge
false
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250
Circle -65536 true false 202 65 90

sick wolf
false
0
Rectangle -7566196 true true 15 105 105 165
Rectangle -7566196 true true 45 90 105 105
Polygon -7566196 true true 60 90 83 44 104 90
Polygon -16777216 true false 67 90 82 59 97 89
Rectangle -1 true false 48 93 59 105
Rectangle -16777216 true false 51 96 55 101
Rectangle -16777216 true false 0 121 15 135
Rectangle -16777216 true false 15 136 60 151
Polygon -1 true false 15 136 23 149 31 136
Polygon -1 true false 30 151 37 136 43 151
Rectangle -7566196 true true 105 120 263 195
Rectangle -7566196 true true 108 195 259 201
Rectangle -7566196 true true 114 201 252 210
Rectangle -7566196 true true 120 210 243 214
Rectangle -7566196 true true 115 114 255 120
Rectangle -7566196 true true 128 108 248 114
Rectangle -7566196 true true 150 105 225 108
Rectangle -7566196 true true 132 214 155 270
Rectangle -7566196 true true 110 260 132 270
Rectangle -7566196 true true 210 214 232 270
Rectangle -7566196 true true 189 260 210 270
Line -7566196 true 263 127 281 155
Line -7566196 true 281 155 281 192
Circle -65536 true false 130 106 106

square
false
0
Rectangle -7566196 true true 48 27 253 222

teddy bear
false
0
Circle -7566196 true true 110 21 81
Circle -7566196 true true 97 7 39
Circle -7566196 true true 171 5 39
Polygon -7566196 true true 133 88 95 117 95 225 119 247 188 248 215 224 215 116 170 91
Circle -7566196 true true 134 238 34
Polygon -7566196 true true 197 113 249 88 261 91 268 106 262 116 205 139 197 113
Polygon -7566196 true true 115 116 66 90 54 93 45 110 50 117 103 145 115 116
Polygon -7566196 true true 104 204 54 233 54 244 63 257 71 256 117 227
Polygon -7566196 true true 194 228 240 255 248 254 260 238 257 231 204 207 194 228
Circle -1 true false 124 41 20
Circle -1 true false 158 42 20
Line -16777216 false 127 75 150 85
Line -16777216 false 151 85 177 72
Polygon -1 true false 152 204 115 167 131 150 150 168 168 152 184 167

thin wedge
false
0
Polygon -7566196 true true 133 20 70 252 127 148 177 245

uu
false
0
Rectangle -7566196 true true 58 44 104 226
Rectangle -7566196 true true 88 178 194 224
Rectangle -7566196 true true 180 44 222 224
Rectangle -7566196 true true 122 74 168 284
Rectangle -7566196 true true 124 240 250 286
Rectangle -7566196 true true 240 76 284 286

wide wedge
false
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

wolf
false
0
Rectangle -7566196 true true 15 105 105 165
Rectangle -7566196 true true 45 90 105 105
Polygon -7566196 true true 60 90 83 44 104 90
Polygon -16777216 true false 67 90 82 59 97 89
Rectangle -1 true false 48 93 59 105
Rectangle -16777216 true false 51 96 55 101
Rectangle -16777216 true false 0 121 15 135
Rectangle -16777216 true false 15 136 60 151
Polygon -1 true false 15 136 23 149 31 136
Polygon -1 true false 30 151 37 136 43 151
Rectangle -7566196 true true 105 120 263 195
Rectangle -7566196 true true 108 195 259 201
Rectangle -7566196 true true 114 201 252 210
Rectangle -7566196 true true 120 210 243 214
Rectangle -7566196 true true 115 114 255 120
Rectangle -7566196 true true 128 108 248 114
Rectangle -7566196 true true 150 105 225 108
Rectangle -7566196 true true 132 214 155 270
Rectangle -7566196 true true 110 260 132 270
Rectangle -7566196 true true 210 214 232 270
Rectangle -7566196 true true 189 260 210 270
Line -7566196 true 263 127 281 155
Line -7566196 true 281 155 281 192

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
